"""
INDIE VTUBER KIT FRAMEWORK (V4 - AUDIO-REACTIVE PHYSICS)
------------------------------------------------------
This version replaces mouth-flapping with a physics-based engine that
reacts to the volume, bass, and treble of your voice for a more
dynamic and modern "wicked cool" effect.
"""

import os
import sys
import io
import time
import numpy as np
import pyaudio
import pygame
from pathlib import Path
from PIL import Image
from rembg import remove
from tqdm import tqdm

# =================================================================
# CONFIGURATION
# =================================================================
class Config:
    INPUT_DIR = './'
    ASSET_DIR = './assets_ready'
    
    # Audio Analysis
    MIC_RATE = 44100
    MIC_CHUNK = 2048 # Larger chunk for better frequency analysis (FFT)
    
    # Define frequency ranges for bass and treble
    FFT_BASS_RANGE = (50, 250)    # Hz
    FFT_TREBLE_RANGE = (2000, 5000) # Hz
    
    # Visuals & Physics
    WINDOW_SIZE = (1920, 1080)
    BG_COLOR = (0, 255, 0) # Green Screen
    GRAVITY = 0.8
    FRICTION = 0.95 # Damping factor for movement
    SMOOTHING = 0.1 # How quickly the avatar lerps to target values
    
    # -- Tweak these "Wicked Cool" values! --
    VOLUME_SCALE_STRENGTH = 0.3    # How much it scales up with volume (0.3 = 30% bigger)
    VOLUME_JUMP_STRENGTH = -20     # Upward force from volume (negative is up)
    BASS_THUMP_STRENGTH = 10       # Downward force from bass
    TREBLE_JITTER_STRENGTH = 15    # Horizontal force from treble
    ROTATION_STRENGTH = 15         # Max tilt in degrees

# =================================================================
# MODULE 1: ASSET PIPELINE (SIMPLIFIED)
# =================================================================
class AssetPipeline:
    def process_asset(self, file_path):
        """
        Processes a single image: Remove BG, crop, and save a single
        high-resolution master avatar file. No talking frames needed.
        """
        try:
            original = Image.open(file_path).convert("RGBA")
            cleaned = remove(original)
            bbox = cleaned.getbbox()
            if not bbox: return
            cropped = cleaned.crop(bbox)
            
            # Save a single, high-quality master image.
            stem = Path(file_path).stem
            out_path = os.path.join(Config.ASSET_DIR, stem)
            os.makedirs(out_path, exist_ok=True)
            
            # We save it at its processed, high-res state.
            # The engine will handle scaling it down to fit the screen.
            cropped.save(os.path.join(out_path, "avatar.png"))
            return True
        except Exception as e:
            print(f" [!] File Error on {Path(file_path).name}: {e}")
            return False

    def run_batch(self):
        print("--- STARTING PHYSICS ASSET PIPELINE ---")
        if not os.path.exists(Config.ASSET_DIR): os.makedirs(Config.ASSET_DIR)
        
        files = []
        valid = {'.png', '.jpg', '.jpeg'}
        for r, d, f in os.walk(Config.INPUT_DIR):
            if "venv" in r or Config.ASSET_DIR in r: continue
            for file in f:
                if Path(file).suffix.lower() in valid and "avatar" not in file:
                    files.append(os.path.join(r, file))
        
        print(f"Found {len(files)} candidates.")
        for f in tqdm(files, desc="Processing Master Avatars"):
            self.process_asset(f)
        print("--- PIPELINE FINISHED ---")


# =================================================================
# MODULE 2: RUNTIME ENGINE (PHYSICS EDITION)
# =================================================================
class AvatarEngine:
    def __init__(self, avatar_name):
        # Pygame & Asset Loading
        pygame.init()
        self.screen = pygame.display.set_mode(Config.WINDOW_SIZE)
        pygame.display.set_caption(f"Physics Avatar: {avatar_name}")
        self.clock = pygame.time.Clock()
        
        avatar_path = os.path.join(Config.ASSET_DIR, avatar_name, "avatar.png")
        self.master_surf = pygame.image.load(avatar_path).convert_alpha()
        
        # Scale master surface to fit 80% of screen height
        target_h = Config.WINDOW_SIZE[1] * 0.8
        scale_ratio = target_h / self.master_surf.get_height()
        target_w = self.master_surf.get_width() * scale_ratio
        self.master_surf = pygame.transform.smoothscale(self.master_surf, (int(target_w), int(target_h)))
        
        # Physics State Variables
        self.pos = pygame.Vector2(Config.WINDOW_SIZE[0] / 2, Config.WINDOW_SIZE[1] / 2)
        self.velocity = pygame.Vector2(0, 0)
        self.current_scale = 1.0
        self.current_angle = 0.0
        
        # Audio Init
        self.p = pyaudio.PyAudio()
        try:
            self.stream = self.p.open(format=pyaudio.paInt16, channels=1, rate=Config.MIC_RATE, input=True, frames_per_buffer=Config.MIC_CHUNK)
        except:
            sys.exit(" [CRITICAL] No Microphone Found.")

    def get_audio_features(self):
        """
        Performs FFT analysis on mic input to get volume, bass, and treble.
        """
        try:
            data = self.stream.read(Config.MIC_CHUNK, exception_on_overflow=False)
            audio_data = np.frombuffer(data, dtype=np.int16)
            
            # 1. Volume (RMS)
            volume = np.sqrt(np.mean(audio_data**2))
            
            # 2. Frequencies (FFT)
            # Perform Fast Fourier Transform
            fft_data = np.fft.rfft(audio_data)
            fft_freqs = np.fft.rfftfreq(len(audio_data), 1.0/Config.MIC_RATE)
            fft_mags = np.abs(fft_data)
            
            # Helper to get energy in a frequency band
            def get_energy(freq_range):
                indices = np.where((fft_freqs >= freq_range[0]) & (fft_freqs <= freq_range[1]))
                return np.sum(fft_mags[indices])

            bass = get_energy(Config.FFT_BASS_RANGE)
            treble = get_energy(Config.FFT_TREBLE_RANGE)

            # Normalize values to be more usable (prevents huge numbers)
            # These thresholds were found by experimentation
            return {
                'volume': np.clip(volume / 2000, 0, 1),
                'bass': np.clip(bass / 200000, 0, 1),
                'treble': np.clip(treble / 200000, 0, 1)
            }
        except:
            return {'volume': 0, 'bass': 0, 'treble': 0}

    def run(self):
        running = True
        while running:
            for event in pygame.event.get():
                if event.type == pygame.QUIT: running = False

            # 1. ANALYZE AUDIO
            audio = self.get_audio_features()

            # 2. UPDATE PHYSICS
            # Apply forces based on audio features
            jump_force = audio['volume'] * Config.VOLUME_JUMP_STRENGTH
            thump_force = audio['bass'] * Config.BASS_THUMP_STRENGTH
            jitter_force = (audio['treble'] * Config.TREBLE_JITTER_STRENGTH) * np.random.choice([-1, 1])

            self.velocity.y += jump_force + thump_force
            self.velocity.x += jitter_force
            
            # Apply constant forces
            self.velocity.y += Config.GRAVITY
            self.velocity *= Config.FRICTION # Slow down over time
            
            # Update position
            self.pos += self.velocity
            
            # Boundary checks (don't fall off screen)
            if self.pos.y > Config.WINDOW_SIZE[1] * 0.7:
                self.pos.y = Config.WINDOW_SIZE[1] * 0.7
                self.velocity.y = 0
            
            # Update scale and rotation with smoothing (lerping)
            target_scale = 1.0 + audio['volume'] * Config.VOLUME_SCALE_STRENGTH
            target_angle = (audio['bass'] - 0.5) * Config.ROTATION_STRENGTH # Bass makes it tilt
            
            self.current_scale += (target_scale - self.current_scale) * Config.SMOOTHING
            self.current_angle += (target_angle - self.current_angle) * Config.SMOOTHING
            
            # 3. RENDER
            self.screen.fill(Config.BG_COLOR)
            
            # Create a transformed copy of the master surface
            transformed_surf = pygame.transform.rotozoom(self.master_surf, self.current_angle, self.current_scale)
            
            # Get the new rectangle and position it
            rect = transformed_surf.get_rect(center = self.pos)
            
            self.screen.blit(transformed_surf, rect)
            pygame.display.flip()
            self.clock.tick(60)
